
// Version b0.03

/*
 * This class contains constants which in some cases are used throughout the 
 * mass equation calculations. The physics constants though are only used to
 * calculate the derived mass element constant, which is one of the 
 * introductory parts of the mass equations. The mass element has type kg.
 *
 * NOTE: I've used the most recent precise values i could find. 
 */
public class Constants {

	
    // h_bar = h/(2 * pi)
    private static double plancksConstant = (6.6260693 * Math.pow(10.0, -34.0)) / (2.0 * Math.PI);

	
    private static double lightVelocity = 2.99792458 * Math.pow(10.0, 8.0);
    
	
    private static double gravitationConstant = 6.6742 * Math.pow(10.0, -11.0);

	
    // Not used yet.
    // magnetic mu = 12.5666370614 * 10-7 == 4* pi* 10^-7
    private static double inductionConstant = Math.PI * 4.0 * Math.pow(10.0,-7.0) ;

	
    // Not used yet.
    // dielectric eps_0 8.854187817 * 10-12 == 1/(mu * c^2)
    private static double influenceConstant = 1.0 / (inductionConstant * lightVelocity * lightVelocity);

	
    // more accurate is 376.730313461 == sqrt(mu/eps_0)== mu*c
    private static double vacuumWaveResistance = (inductionConstant * lightVelocity);

    
    public static double geometricalConstant = (1.0 + Math.sqrt(5.0))/2.0;


    /*
     * The Mass Constant - EQUATION VI (p4, 1982 formula) 
     */
    public static double massElement =	
	(MathL.root(Math.PI, 4.0) * 
	 MathL.root(3.0 * Math.PI * gravitationConstant * plancksConstant, 3.0) *
	 Math.sqrt(plancksConstant / (3.0 * lightVelocity * gravitationConstant)));


    /* Inverse fine-structure constant - Note this is "alpha == alpha(+)^(-1)" !
     * alpha - middle of page (p4, 1982 formula)
     */
    public static double alpha = 1.0 / 137.03599911;


    /*
     * eta - constant version of eta in EQUATIONS V (p3, 1982 formula) and VII
     *
     * NOTE: The non-constant versions can be found in the EQ class.
     */
    public static double eta = Math.PI / MathL.root((Math.pow(Math.PI, 4.0) + 4.0), 4.0);


    /*
     * theta - constant version of theta in EQUATIONS V (p3, 1982 formula)
     *
     * NOTE: The non-constant versions can be found in the EQ class.
     */
    public static double theta = (5.0 * eta) + (2.0 * Math.sqrt(eta)) + 1.0;


    // HD: just before eq. VII.
    private static double t = 1.0 - ((2.0/3.0) * geometricalConstant * 
				    Math.pow(eta, 2) * (1.0 - Math.sqrt(eta)));


    /*
     * alpha+ - one of EQUATIONS VIII (p4, 1982 formula)
     */
    public static double alphaPlus = t * (1.0 / ( Math.pow(eta, 2.0) * MathL.root(eta, 3.0) )) - 1.0 ;


    /*
     * alpha- - one of EQUATIONS VIII (p4, 1982 formula)
     */
    public static double alphaMinus = t * (1.0 /(eta * MathL.root(eta, 3.0) )) - 1.0;
	

    /*
     * alpha+ - one of EQUATIONS B4 (p11, 1982 formula)
     */
    public static double altAlphaPlus = 
		(Math.pow(eta, (1.0/6.0))/Math.pow(eta, 2.0) *
		(1.0 - theta*Math.pow((2.0*(1.0-Math.sqrt(eta)))/(eta*(1.0+Math.sqrt(eta))), 2.0)*
		 Math.sqrt(2.0*eta)) - 1.0);

	
    /*
     * alpha- - one of EQUATIONS B4 (p11, 1982 formula)
     */
    public static double altAlphaMinus = ((altAlphaPlus+1.0)*eta) - 1.0;

}
