
// Version b0.03

/*
 * This class is the main entry point for the mass equation given as M, EQUATION
 * B3 (p11, 1989 formula). The parameter mu in the front of the mass equation is 
 * given in kg thus the mass equation result is in kg (since no other values with 
 * type occur in the formula). The mass is converted to MeV before being returned.
 */
public class MainFunctions {

    private static final double KILO_PER_UNIT = 1.660540*Math.pow(10, -27);
    private static final double MEV_PER_UNIT = 931.4943;

    /*
     * The Mass Equation, M - EQUATION B3 (p11, 1989 formula)
     *
     * The parameters are those found in Tabel I (p2, Selected Results) for particles:
     *         k - column 1
     *         q - absolute value of column 8 (epsilon_q)
     *         n - column 2
     *         m - column 3
     *         p - column 4
     *     sigma - column 5
     *         P - column 6
     *         Q - column 7
     *     kappa - column 10
     * epsilon_q - column 8
     */
    public static double massEquation(double k, double q,
				      double n, double m, double p, double sigma,
				      double P, double Q,
				      double kappa, double epsilon_q) {

	// Result of mass equation (in kg)
	double M = (Constants.massElement *
		    Constants.alphaPlus *
		    ((G(k, q) + 
		      S(n, m, p, sigma, k, q) + 
		      F(n, m, p, sigma, P, Q, k, q, kappa, epsilon_q) + 
		      phi(P, Q, k, q)) +
		     (4*q*Constants.alphaMinus)));

	// Conversion to MeV
	return (M/KILO_PER_UNIT)*MEV_PER_UNIT;
    }

    /*
     * Alternative mass equation using Particle as input parameter
     */
    public static double massEquation(Particle particle, double n, double m, double p, double sigma) {

	// Result of mass equation (in kg)
	double M = (Constants.massElement *
		    Constants.alphaPlus *
		    ((G(particle.k, particle.q) + 
		      S(n, m, p, sigma, particle.k, particle.q) + 
		      F(n, m, p, sigma, particle.P, particle.Q, particle.k, particle.q, particle.kappa, particle.qx) + 
		      phi(particle.P, particle.Q, particle.k, particle.q)) +
		     (4*particle.q*Constants.alphaMinus)));

	// Conversion to MeV
	return (M/KILO_PER_UNIT)*MEV_PER_UNIT;
    }

    /*
     * G - EQUATION (G underline) XI (p5, 1982 formula)
     */ 
    private static double G(double k, double q) {

	return ((Math.pow(QN.Q1(k), 2)*Math.pow(1+QN.Q1(k), 2)*QN.N1(k, q)) +
		(QN.Q2(k)*((2*Math.pow(QN.Q2(k), 2))+(3*QN.Q2(k))+1)*QN.N2(k, q)) +
		(QN.Q3(k)*(1+QN.Q3(k))*QN.N3(k, q)) +
		(4*QN.Q4(k)));
    }

    /*
     * S - EQUATION (K) XI (p5, 1982 formula)
     */ 
    private static double S(double n, double m, double p, double sigma, double k, double q) {

	return ((Math.pow(n, 2)*Math.pow(1+n, 2)*QN.N1(k, q)) +
		(m*((2*Math.pow(m, 2))+(3*m)+1)*QN.N2(k, q)) +
		(p*(1+p)*QN.N3(k, q)) +
		(4*sigma));
    }

    /*
     * F1982 (+ phi) - EQUATION (H) XI (p5, 1982 formula)
     *
     * NOTE: Their are major problems with the self-couplings function.
     * Look in class SelfCouplingsFunction for details.
     */ 
    private static double F1982(double n, double m, double p, double sigma, 
			   double P, double Q, 
			   double k, double q, 
			   double kappa, double epsilon_q) {

	return (((2*n*QN.Q1(k))*(1+(3*(n + QN.Q1(k) + (n*QN.Q1(k))))+(2*(Math.pow(n, 2)+Math.pow(QN.Q1(k), 2))))*QN.N1(k, q)) +
		(6*m*QN.Q2(k)*(1+m+QN.Q2(k))*QN.N2(k, q)) +
		(2*p*QN.Q3(k)*QN.N3(k, q))
		- (SelfCouplingFunction.phi(k, q, p, sigma, P, Q, kappa, epsilon_q)));
    }

    /*
     * F1989 - EQUATION B5 (p11, 1989 formula)
     *
     * NOTE: Their are major problems with the self-couplings function.
     * Look in class SelfCouplingsFunction for details.
     */ 
    private static double F1989(double n, double m, double p, double sigma, 
			   double P, double Q, 
			   double k, double q, 
			   double kappa, double epsilon_q) {

	return (((2*n*QN.Q1(k))*(1+(3*(n + QN.Q1(k) + (n*QN.Q1(k))))+(2*(Math.pow(n, 2)+Math.pow(QN.Q1(k), 2))))) +
		(6*m*QN.Q2(k)*(1+m+QN.Q2(k))*QN.N2(k, q)) +
		(2*p*QN.Q3(k)*QN.N3(k, q))
		- (SelfCouplingFunction.phi(k, q, p, sigma, P, Q, kappa, epsilon_q)));
    }

    // There is a slight difference between F in the 1982 note compared to the 1989 note
    // besides the addition of phi. The difference concernes a factor N1 after the first 
    // bracket which seems to have been missplaced in the 1989 version.
    private static double F(double n, double m, double p, double sigma, 
			    double P, double Q, 
			    double k, double q, 
			    double kappa, double epsilon_q) {

	return F1982(n, m, p, sigma, 
		      P, Q, 
		      k, q, 
		      kappa, epsilon_q);
    }


    /*
     * phi - EQUATION B6 (p12, 1989 formula)
     */ 
    private static double phi(double P, double Q, double k, double q) {

	return (P*Math.pow(-1, P+Q)*((P+Q)*QN.N5(k, q)))+(Q*(P+1)*QN.N6(k, q));
    }
}
