
/*
 * Objects of this class represent particles. Particles are uniquely 
 * given by the parameters k, P, Q, kappa and x. From these values the 
 * structure distributor C, the electrical charge quantum number qx and 
 * the electrical charge amount q is calculated.
 *
 * NOTE: The value x for particles can be choosen amoung those values x
 * which fulfill 0 <= x <= P for any combination of the 4 other input 
 * values.
 *
 * NOTE: Major probems:
 *
 *    1) Which quantum number equation should be used, 1982 or 1989?
 *       Currently 1989 is used.
 *
 *    2) Should Time-helicity be true (+1) or false (-1)? see p2 1989 
 *       note.
 */
public class Particle {

    // Time helicity - p3 1982 (seems it's supposed to be true ie. '+1'?)
    public static final boolean TIME_HELICITY = true;

    public final int k;

    // Double Isospin
    public final int P;

    // Double Space-spin
    public final int Q;

    // Doblet Number
    public final int kappa;

    // Component of the Isospin-multiplet
    public final int x;

   // Structure Distributor
    public final int C;

    // Quantum Number (Electrical Charge)
    public final int qx;

   // Electrical Charge Amount
    public final int q;

    public Particle(int k, int P, int Q, int kappa, int x) throws Exception {

	// Make sure input parameters are ligit
	if(checkParameterConstraints(k, P, Q, kappa, x)) {
	    throw new Exception("Paramter Constraints Violated");
	}

	this.k = k;
	this.P = P;
	this.Q = Q;
	this.kappa = kappa;
	this.x = x;

	this.C = getStructureDistributor(TIME_HELICITY);
	this.qx = getQuantumNumber(TIME_HELICITY);
	this.q = getChargeAmount();
    }

    /*
     * C - one of EQUATIONS I (p2, 1982 formula)
     */
    private int getStructureDistributor(boolean timeHelicity) {

	double alpha_P = Math.PI * Q * (kappa + MathL.comb(P, 2));

	double alpha_Q = Math.PI * Q * (Q + MathL.comb(P, 2));

	double epsilon_P;
	double epsilon_Q;
	if(timeHelicity) {

	    epsilon_P = Math.cos(alpha_P);
	    epsilon_Q = Math.cos(alpha_Q);
	}
	else {

	    epsilon_P = -Math.cos(alpha_P);
	    epsilon_Q = -Math.cos(alpha_Q);
	}

	return (int)((2 * ((P*epsilon_P) + (Q*epsilon_Q)) * (k - 1 + kappa)) / ((1 + kappa) * k));

    }

    /*
     * qx - EQUATION B2 (p11, 1989 formula)
     */
    private int getQuantumNumber1989(boolean timeHelicity) {

	if(timeHelicity) {

	    return (int)((1.0/2.0) * (((P-(2*x)+2) * (1 - (kappa*Q*(2-k)))) + (k-1-((1+kappa)*Q*(2-k))) + C));
	}
	else {

	    return (int)((1.0/2.0) * (((P-(2*x)+2) * (1 - (kappa*Q*(2-k)))) - (k-1-((1+kappa)*Q*(2-k))) + C));
	}
    }

    /*
     * qx - one of EQUATIONS II (p3, 1982 formula)
     */
    private int getQuantumNumber1982(boolean timeHelicity) {

	if(timeHelicity) {

	    return (int)((1.0/2.0) * (((P-(2*x)) * (1 - (kappa*Q*(2-k)))) + (k-1-((1+kappa)*Q*(2-k))) + C));
	}
	else {

	    return (int)((1.0/2.0) * (((P-(2*x)) * (1 - (kappa*Q*(2-k)))) - (k-1-((1+kappa)*Q*(2-k))) + C));
	}
    }

    /*
     * Which quantum number equation should be used.... 1982 or 1989?
     * 1982 - returns identical results to Selected Results note
     * 1989 - returns inccorect results to Selected Results note
     */
    private int getQuantumNumber(boolean timeHelicity) {

	return getQuantumNumber1989(timeHelicity);
    }

    /*
     * q - one of EQUATIONS II (p3, 1982 formula)
     */
    private int getChargeAmount() {

	return Math.abs(qx);
    }

    /*
     * Lets make sure that constructor parameters are ligit
     * NOTE: Since the global vars are final they cannot be
     * changed once they pass this check
     */
    private boolean checkParameterConstraints(int k, int P, int Q, int kappa, int x) {

	boolean volation = false;

	if (k < 1 || 2 < k)
	    volation = true;

	if (P < 0 || 3 < P)
	    volation = true;

	if (Q < 0 || 3 < Q)
	    volation = true;

	if (kappa < 0 || 1 < kappa)
	    volation = true;

	if (x < 0 || P < x)
	    volation = true;

	return volation;
    }

    public String toString() {

	String type = "";
	if (k == 1 && Q == 1) {

	    type = "Lepton";
	}
	else if (k == 1 && Q == 0) {

	    type = "Messon";
	}
	else {

	    type = "Baryon";
	}

	return type;
    }
}
