
// Version b0.03

/*
 * This class deals with equations related to determining the value of the Quantum 
 * Number EQUATIONS IX-X (p5, 1982 formula) and B8-B14 (p12, 1989 formula). Note that
 * equation N3 is the one given in the 1989 formula. 
 */
public class QN {

    /*
     * eta - one of EQUATIONS V (p3, 1982 formula)
     *
     * NOTE: eta(1,0) = eta according to the 1989 formula. The constant version 
     * of the function eta, eta(1,0) is given in the Constants class.
     */
    public static double eta(double k, double q) {

	return Math.PI/Math.pow(Math.pow(Math.PI, 4.0)+((4.0+k)*Math.pow(q, 4.0)), (1.0)/(4.0));
    }

    /*
     * theta - one of EQUATIONS V (p3, 1982 formula)
     *
     * NOTE: theta(1,0) = theta according to the 1989 formula. The constant version 
     * of the function theta, theta(1,0) is given in the Constants class.
     */
    public static double theta(double k, double q) {

	return (5*eta(k, q)) + (2*Math.sqrt(eta(k, q))) + 1;
    } 

    /*
     * N1 - one of EQUATIONS IX (p5, 1982 formula)
     */
    public static double N1(double k, double q) {

	return 0.5*(1+Math.sqrt(eta(k,q)));
    }

    /*
     * N2 - one of EQUATIONS IX (p5, 1982 formula)
     */
    public static double N2(double k, double q) {

	return (2.0/3.0)*(1/eta(k,q));
    }


    //return (7.297352568 * Math.pow(10.0,-3.0));
    private static double fineStructureConstant = Constants.alpha;


    // (1+sqrt(5))/2.0   E_ docum 1982, p4
    // return (1.61803399);
    private static double thetaConstant = Constants.geometricalConstant;


    // Document D, p67, (9.9)
    // D_ doc (seems good :) - we are only off for q=2 and N3(2,2)
    private static double H(double k, double q) {
        
	return ((fineStructureConstant /3.0 ) * (1.0 + Math.sqrt(eta(k, q))) * 
		Math.pow( (thetaConstant / (eta(k,q)*eta(k,q)) ), (2.0*k + 1.0) ) *
		Math.pow(eta(k, q),3.0));
    }

    // Document D, p67, (9.9)
    // D_ doc  better? has to be right
    private static double G(double k, double q) {
       
	return ((eta(1,1)/ (Math.E * eta(k,q))) * 
		Math.pow((2.0 * thetaConstant * eta(k,q) ),k) *
		Math.pow( ((1.0 - Math.sqrt(eta(k,q))) / (1.0 + Math.sqrt(eta(k,q)))) ,2.0));
    }

    /*
     * N3 - EQUATION (9.9) (p67)
     *   http://www.heim-theory.com/downloads_pw/D_Zur_Herleitung_Der_Heimschen_Massenformel.pdf
     */
    private static double N3b(double k, double q) {

       // E_ doc don't forget times 2.0
       return 2.0 * ((Math.pow (Math.E,(k-1))/k) - 
		     q *(H(k,q) + G(k,q))); 
    }

    /*
     * N3 - EQUATION B8 (p12, 1989 formula)
     * From http://www.heim-theory.com/downloads/F_Heims_Mass_Formula_1989.pdf
     */
    private static double N3a(double k, double q) {

	double u = 2.0 * Math.PI*Math.E;

	return (Math.pow(Math.E, (k-1)*
			 (1-(Math.PI*((1-eta(k, q))/(1+Math.sqrt(eta(1, q))))*
			     (1-(u*((eta(1, q))/(theta(1, q)))*
				 (1-(Constants.alphaMinus/Constants.alphaPlus))*
				 Math.pow((1-Math.sqrt(Constants.eta)), 2))))) -
			 ((2/(3*Math.PI*Math.E))*
			  Math.pow((1-Math.sqrt(Constants.eta)), 2)*
			  ((6.0*Math.pow(Math.PI, 2)*(Math.pow(Math.E, 2)/Constants.theta)*
			    ((1+Math.sqrt(eta(1, q)))/(1-Constants.eta)))-1)))/(k/2.0));
    }

    // There is currently a conflict regarding which version of N3 is the correct
    // one, this method simply serves as a quick shifter.
    public static double N3(double k, double q) {
      return N3b(k,q);
    }

    /*
     * N4 - EQUATION B9 (p12, 1989 formula)
     */
    public static double N4(double k, double q) {

	return (4.0/k)*(1.0+(q*(k-1.0)));
    }

    /*
     * A - EQUATION B11 (p12, 1989 formula)
     */
    private static double A() {

	return ((8.0/Constants.eta)*
		(1.0 - (Constants.alphaMinus/Constants.alphaPlus) )*
		(1.0 - ((3.0*Constants.eta)/4.0)) );
    }

    /*
     * N - EQUATION B12 (p12, 1989 formula)
     */
    private static double N(double k) {

	return (Q1(k)+Q2(k)+Q3(k)+Q4(k)+(k*(Math.pow(-1, k))*Math.pow(2, Math.pow(k, 2)-1)));
    }

    /*
     * N' - EQUATION B14 (p12, 1989 formula)
     */
    private static double diffN(double k) {

	return (Q1(k)+Q2(k)+Q3(k)+Q4(k)-(2*k)-1);
    }

    /*
     * N5 - EQUATION B10 (p12, 1989 formula)
     */
    public static double N5(double k, double q) {

	return (A()*(1.0+
                 ( k*(k - 1.0)*
			 Math.pow(2.0, Math.pow(k, 2.0)+3.0)*
                   N(k)*
			 A()* Math.pow((1-Math.sqrt(eta(k, q)))/(1+Math.sqrt(eta(k, q))), 2) 
                  ) ));
    }

    /*
     * N6 - EQUATION B13 (p12, 1989 formula)
     */
    public static double N6(double k, double q) {

	return (((2*k)/(Math.PI*Math.E*Constants.theta))*
		((Math.sqrt(k)*(Math.pow(k, 2)-1)*(N(k)/Math.sqrt(eta(k, 1)))*
		  (q-((1-q)*(diffN(k)/(Q1(k)*Math.sqrt(eta(k, 1)))))))+Math.pow(-1, k+1))*
		Constants.eta*(1-(Constants.alphaMinus/Constants.alphaPlus))*
		Math.pow(4*((1-Math.sqrt(Constants.eta))/(1+Math.sqrt(Constants.eta))), 2)*Q4(k));
    }


    /*
     * Small Function needed for the below equations
     */
    private static double s(double k) {

	return Math.pow(k, 2) + 1;
    }

    /*
     * Q1 - one of EQUATIONS X (p5, 1982 formula)
     * 
     * NOTE: in the 1989 formula this is refered to as Qn
    */
    public static double Q1(double k) {
	
	return 3 * Math.pow(2, s(k)-2);
    }

    /*
     * Q2 - one of EQUATIONS X (p5, 1982 formula)
     * 
     * NOTE: in the 1989 formula this is refered to as Qm
     */
    public static double Q2(double k) {
	
	return Math.pow(2, s(k))-1;
    }

    /*
     * Q3 - one of EQUATIONS X (p5, 1982 formula)
     * 
     * NOTE: in the 1989 formula this is refered to as Qp
     */
    public static double Q3(double k) {
	
	return Math.pow(2, s(k))+(2*Math.pow(-1, k));
    }

    /*
     * Q4 - one of EQUATIONS X (p5, 1982 formula)
     * 
     * NOTE: in the 1989 formula this is refered to as Qsigma
     */
    public static double Q4(double k) {
	
	return Math.pow(2, s(k)-1)-1;
    }
}
