
// Version b0.03

/*
 * This class deals with all equations related to determining the value of the so 
 * called 'self-couplings function' phi given as EQUATION B7 (p12, 1989 formula).
 * It is needed to evaluate EQUATION B5 (p11, 1989 formula).
 * 
 * The main sub-equations of the self-couplings function are EQUATIONS B22-B28 
 * (p13, 1989 formula) as well as U given just below phi EQUATION B7. Also a1,
 * a2, and a3 EQUATIONS B29-B31 (p14, 1989 formula) are needed to evaluate x
 * EQUATION B27 (p13, 1989 formula).
 *
 * NOTE: Minor probems:
 *
 *    1) a1, a2 and a3 return incorrect values for a few particals Tabel VIII
 *       (p10, Selected Results).
 *
 *    2) A returns incorrect values as opposed to those found in Tabel VI 
 *       (p9, Selected Results).
 */
public class SelfCouplingFunction {

    /*
     * Self Couplings Function phi - EQUATION B7 (p12, 1989 formula)
     */
    public static double phi(double k, double q, double p, double sigma, double P, double Q, double kappa, double epsilon_q) {
	
	// NOTE: phi is solely incorrect at this point because W is incorrect!

	return ((((QN.N4(k, q)*Math.pow(p, 2))/(1+Math.pow(p, 2)))*
		 ((sigma+QN.Q4(k))/Math.sqrt(1+Math.pow(sigma, 2)))*
		 (MathL.root(2, 4.0)-(4*B(k)*U(k, q, P, Q, kappa)*Math.pow(W(k, q, P, Q, kappa, epsilon_q), -1)))) +
		(P*Math.pow(P-2, 2)*(1 + (kappa*((1-q)/(2*Constants.alpha*Constants.theta))))*
		 Math.pow(Math.PI/Math.E, 2)*Math.sqrt(QN.eta(2, 1))*(QN.Q2(k)-QN.Q1(k))) -
		 ((P + 1)*(MathL.comb((int)Q, 3)/Constants.alpha)));
    }

    /*
     * U - Given just below EQUATION B7 (p12, 1989 formula) or as EQUATION B50 (p17, 1989 formula)
     */
    private static double U(double k, double q, double P, double Q, double kappa) {

	return (Math.pow(2, k+P+Q+kappa)*
		(Math.pow(P, 2)+((3.0/2.0)*(P-Q))+(P*(1-q))+(4*kappa*B(k)*((1-Q)/(3-(2*q))))+
		 ((k-1)*(P+(2*Q)-(4*Math.PI*(P-Q)*((1-q)/MathL.root(2, 4.0))))))*Math.pow(QN.eta(k, q), -2));
    }

    /*
     * W - EQUATION B22 (p13, 1989 formula)
     */
    public static double W(double k, double q, double P, double Q, double kappa, double epsilon_q) {

	// NOTE: the values for function W on tables VIII do not correlate...
	// W is the reason that phi is impricise!

	return ((A(k)*Math.pow(Math.E, x(k, q, P, Q, kappa, epsilon_q))*Math.pow(1-Constants.eta, L(k, Q, kappa))) +
		((P-Q)*(1-MathL.comb((int)P, 2))*(1-MathL.comb((int)Q, 3))*Math.pow(1-Math.sqrt(Constants.eta), 2)*Math.sqrt(2)));
    }

    /*
     * A - EQUATION B23 (p13, 1989 formula)
     */
    public static double A(double k) {

	// NOTE: the values for function A on tables VI do not correlate...
	
	// Is this a problem with function g?
	
	// Interresting that A(1) has the exact same decimal figures 
	// as in the table but incorrect overall value

    	return 8*g(k)*H(k)*Math.pow(2-k+(8*H(k)*(k-1)), -1);
    }

    /* This is an alternative function for A which uses the table VI values
     * instead since there seems to be some problem generating the same results
    */
    public static double altA(double k) {

	if (k == 1) return 2787.59025432;
	else return 14727.57867072;
    }

    /*
     * H - EQUATION B24 (p13, 1989 formula)
     *
     * Relies on the Quantum Number equations, these are in the QN class.
     */
    public static double H(double k) {

	return QN.Q1(k)+QN.Q2(k)+QN.Q3(k)+QN.Q4(k);
    }

    /*
     * g - EQUATION B25 (p13, 1989 formula)
     *
     * Relies on the Quantum Number equations, these are in the QN class.
     */
    private static double g(double k) {

	// NOTE: the values for function A on tables VI do not correlate...
	// it appears as if the problem is with this function

	return (Math.pow(QN.Q1(k), 2) +
		Math.pow(QN.Q2(k), 2) +
		((Math.pow(QN.Q3(k), 2)/k)*Math.pow(Math.E, k-1)) +
		Math.pow(Math.E, ((1-(2*k))/3)) -
		(H(k)*(k-1)));
    }

    /*
     * L - EQUATION B26 (p13, 1989 formula)
     */
    private static double L(double k, double Q, double kappa) {

        return (1-kappa)*Q*(2-k);
    }

    /*
     * x - EQUATION B27 (p13, 1989 formula)
     */
    private static double x(double k, double q, double P, double Q, double kappa, double epsilon_q) {

	return (((1-Q-MathL.comb((int)P, 2))*(2-k)) + 
		((1.0/(4.0*B(k)))*
		 (a1(k, q, P, Q, kappa)+((Math.pow(k, 3)/(4*H(k)))*
					 (a2(k, q, P, Q, kappa, epsilon_q)+
					 (a3(k, q, P, Q, kappa, epsilon_q)/(4*B(k))))))));

    }

    /*
     * B - EQUATION B28 (p13, 1989 formula)
     */
    public static double B(double k) {

	return 3*H(k)*Math.pow(Math.pow(k, 2)*((2*k)-1), -1);
    }

    // a1, a2 and a3

    /*
     * a1 - EQUATION B30 (p14, 1989 formula)
     */
    public static double a1(double k, double q, double P, double Q, double kappa) {

	return (1 + B(k) + (k*(Math.pow(Q, 2)+1)*MathL.comb((int)Q, 3)) -
		(kappa*((B(k)-1)*(2-k)-(3*(H(k)-(2*(1+q)))*(P-Q))+1)) -
		((1-kappa)*
		 ((((3*(2-q)*MathL.comb((int)P, 2))-(Q*((3*(P+Q))+q)))*(2-k))+
		  (((k*(P+1)*MathL.comb((int)P, 2))+
		    (((1+((B(k)/k)*(k+P-Q)))*(1-MathL.comb((int)P, 2))*(1-MathL.comb((int)Q, 3)))-
		     (q*(1-q)*MathL.comb((int)Q, 3))))*(k-1)))));
    }

    /*
     * a2 - EQUATION B29 (p14, 1989 formula)
     */
    public static double a2(double k, double q, double P, double Q, double kappa, double epsilon_q) {

	return ((B(k)*(1-(MathL.comb((int)Q, 3)*(1-MathL.comb((int)P, 3))))) +
		(6.0/k)-(kappa*(((Q/2.0)*(B(k)-(7*k)))-(((3*q)-1)*(k-1))+((1.0/2.0)*(P-Q)*(4+((B(k)+1)*(1-q)))))) -
		((1-kappa)*((((P*((B(k)/2.0)+2+q))-(Q*((B(k)/2.0)+1-(4*(1+(4*q))))))*(2-k))+
			    ((((1.0/4.0)*(B(k)-2)*(1+((3.0/2.0)*(P-Q))))-
			     ((B(k)/2.0)*(1-q))-(MathL.comb((int)P, 2)*(((((1.0/2.0)*(B(k)+q-epsilon_q))+(3*epsilon_q))*(2-epsilon_q)) -
							     ((1.0/4.0)*(B(k)+2)*(1-q)))))*((1-MathL.comb((int)Q, 3))*(k-1))) -
			    (MathL.comb((int)P, 3)*((2*(1+epsilon_q))+
						    ((1.0/2.0)*(2.0-q)*((3*(1.0-q))+epsilon_q-q))-
						    ((q/4.0)*(1-q)*(B(k)-4))-
						    ((1.0/4.0)*(B(k)-2))+
						    ((B(k)/2.0)*(1-q)))))));
    }

    /*
     * a3 - EQUATION B31 (p14, 1989 formula)
     */
    public static double a3(double k, double q, double P, double Q, double kappa, double epsilon_q) {

	// NOTE: The parenthesis starting at '((q*(B(k)/2.0)*(B(k)+(2*(P-Q))))+' never
	// ends in the given equation. Asuming it ends at the end of the given sub-equation above

	double y2B = ((kappa*(((Math.sqrt(Constants.eta)/k)*
			       ((4*(2-Math.sqrt(Constants.eta)))-(Math.PI*Math.E*(1-Constants.eta)*Math.sqrt(Constants.eta)))*
			       (k+(Math.E*Math.sqrt(Constants.eta)*(k-1))))+
			      (((5*(1-q))/((2*k)+Math.pow(-1, k)))*((4*B(k))+P+Q)))) +
		      ((1-kappa)*(((P-1)*(P-2)*(((2.0/Math.pow(k, 2))*(H(k)+2))+((2-k)/(2*Math.PI))))+
				  (MathL.comb((int)P, 2)*(1-MathL.comb((int)Q, 3))*
				   ((q*(B(k)/2.0)*(B(k)+(2*(P-Q))))))+
				    (((P*(P+2)*B(k))+Math.pow(P+1, 2)-
				      (q*(1+epsilon_q)*((k*(Math.pow(P, 2)+1)*(B(k)+2))+((1.0/4.0)*(Math.pow(P, 2)+P+1))))-
				      (q*(1-epsilon_q)*(B(k)+Math.pow(P, 2)+1)))*(k-1))+
				    ((((P-Q)*(H(k)+2))+
				      (P*((5*B(k)*(1+q)*Q)+
					  (k*(k-1)*((k*Math.pow(P+Q, 2)*(H(k)+(3*k)+1)*(1.0-q))-((1.0/2.0)*(B(k)+(6*k))))))))*
				     (1-MathL.comb((int)P, 2))*(1-MathL.comb((int)Q, 3)))+
				    (MathL.comb((int)P, 3)*(2-q)*Q*((epsilon_q*(B(k)+(2*Q)+1))+
								    ((q/(2*k))*(1-epsilon_q)*((2*k)+1))+
								    ((1-q)*(Math.pow(Q, 2)+1+(2*B(k)))))))));

	double y = y2B / (2*B(k));

	return (4*B(k)*(y/(y+1))) - Math.pow(B(k)+4, -1);
    }

}
